
#include "playback.h"
#include "sdcard/sd.h"
#include "sdcard/ff.h"
#include "wav.h"
#include "Timer.h"
#include "DAC.h"
#include "p33Fxxxx.h"
#include <string.h>
#include <stdlib.h>

unsigned char/*playback_state_t*/ playback_state;
unsigned char/*playback_error_t*/ last_playback_error;
unsigned char/*playback_order_t*/ desired_playback_order, current_playback_order;

#define INIT_ATTEMPTS 3
#define INIT_DELAY    200
#define SORT_ENTRIES     256
#define SORT_ENTRY_SIZE (DAC_BUFFER_PAGES*512/SORT_ENTRIES)
#define MAX_FOLDER_DEPTH 8

extern CARD_INFO cardInfo;
static FATFS fs;
static DIR cur_pos;
static FILINFO cur_file_info;
static FIL cur_file;
static unsigned long wav_data_start;
static unsigned long wav_bytes_remaining;
static unsigned long wav_data_size;
static unsigned short wav_data_align, wav_sample_rate;
static unsigned char wav_num_channels, wav_bytes_per_sample, wav_dummy, muted;
static unsigned char transitioning_to_next_file, order_pos, play_just_one, dont_repeat_all;
static unsigned char order[SORT_ENTRIES];
static unsigned short order_num_files;
unsigned short g_volume = 256;

static unsigned char cur_folder_level;
//static unsigned short folder_seeds[MAX_FOLDER_DEPTH];
static char cur_path[_MAX_LFN+5];


static inline bool set_sample_rate(unsigned short rate) {
  if( rate )
    return dac_set_sample_rate(rate);
  else
    return false;
}

unsigned char do_open_wav() {
	char wavbuf[256];
	unsigned int nRead;
	WAVHeader header;

	if( f_read(&cur_file, (BYTE*)wavbuf, sizeof(wavbuf), &nRead) != FR_OK ) {
		playback_state = error;
		last_playback_error = file_read_error;
		return last_playback_error;
	}
	WAVinitHeader(&header);
	if( !WAVreadWAVHeader((BYTE*)wavbuf, &header, nRead) ) {
		playback_state = error;
		last_playback_error = invalid_wav_file;
		return last_playback_error;
	}
	if( header.fmtHeader.audioFormat != 1 || (header.fmtHeader.numChannels != 1 && header.fmtHeader.numChannels != 2) ||
        header.fmtHeader.bitsPerSample != 16 || !set_sample_rate(header.fmtHeader.sampleRate) ) {
		playback_state = error;
		last_playback_error = invalid_wav_format;
		return last_playback_error;
	}
	wav_data_start = header.dataHeader.dataPtr - (BYTE*)wavbuf;
	if( f_lseek(&cur_file, wav_data_start) != FR_OK ) {
		playback_state = error;
		last_playback_error = invalid_wav_file;
		return last_playback_error;
	}
	wav_data_size = wav_bytes_remaining = header.dataHeader.data.chunkSize;
	wav_data_align = 512 - ((512 - (wav_data_start&511)) & ~(header.fmtHeader.numChannels * header.fmtHeader.bitsPerSample / 8 - 1));
	if( header.fmtHeader.numChannels == 1 && wav_data_align > 256 )
		wav_data_align -= 256;
	wav_sample_rate = header.fmtHeader.sampleRate;
	wav_num_channels = header.fmtHeader.numChannels;
	wav_bytes_per_sample = header.fmtHeader.numChannels * header.fmtHeader.bitsPerSample / 8;
	wav_dummy = 0;
	return 0;
}

static unsigned char open_folder() {
	int len = strlen(cur_path);
	int flen = strlen(cur_file_info.fname);
	if( len + flen + (len > 3 ? 1 : 0) < sizeof(cur_path) ) {
        if( len > 3 )
            cur_path[len++] = '\\';
		strcpy(cur_path+len, cur_file_info.fname);
		++cur_folder_level;
		current_playback_order = directory;
	} else {
		playback_state = error;
		last_playback_error = max_depth_exceeded;
	}
	return last_playback_error;
}

static unsigned char goto_first_wav_file();
static unsigned char update_playback_order(unsigned char retain_old_pos);
static unsigned char goto_last_wav_file();
static unsigned char goto_last_dir_or_first_wav_file();

unsigned char open_current_file() {
	char path[_MAX_LFN+5];
   	int len = strlen(cur_path);
	strcpy(path, cur_path);
	if( path[len-1] != '\\' )
		path[len++] = '\\';
	strncpy(path+len, cur_file_info.fname, sizeof(path)-len);
    if( strlen(cur_file_info.fname) < 12 ) {
      len += strlen(cur_file_info.fname);
	  strncpy(path+len, cur_file_info.fname+8, sizeof(path)-len);
    }
	path[sizeof(path)-1] = '\0';
	return f_open(&cur_file, path, FA_READ|FA_OPEN_EXISTING) == FR_OK;
}

static unsigned char open_wav_file(bool bReverse, bool bReverseDir) {
	if( (cur_file_info.fattrib&AM_DIR) ) {
		if( open_folder() != ok )
			return last_playback_error;
		if( bReverse )
			return goto_last_wav_file();
        if( bReverseDir )
			return goto_last_dir_or_first_wav_file();
		else
			return goto_first_wav_file();
	} else {
		if( !open_current_file() ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( do_open_wav() )
			return last_playback_error;

		playback_state = stopped;
		last_playback_error = ok;
		return last_playback_error;
	}
}

static void insert_shuffle_index(unsigned short pos, unsigned short num, unsigned char which) {
	while( pos <= num ) {
		unsigned char old = order[pos];
		order[pos] = which;
		which = old;
		++pos;
	}
}

unsigned long gerhard_random;
unsigned short get_gerhard_random() {
	gerhard_random = (gerhard_random * 32719 + 3) % 32749;
	return gerhard_random;
}
void seed_gerhard_random(unsigned short seed) {
	gerhard_random += seed;
	get_gerhard_random();
	gerhard_random += seed;
}

static unsigned char goto_file_index(unsigned short index) {
	const char* p;
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( ((cur_file_info.fattrib&AM_DIR) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( index-- == 0 )
				return last_playback_error;
		}
	} while( cur_pos.sect != 0 );

	// this should not happen as we should only be passing in the index of files that exist
	playback_state = error;
	last_playback_error = no_wav_files;
	return last_playback_error;
}

static unsigned char find_file_index(unsigned char file_index) {
	unsigned short i;
	for( i = 0; i < order_num_files; ++i )
		if( order[i] == file_index )
			return i;
	return 0;
}

static unsigned short GetRandomSeed() {
  return TMR1;
}

static unsigned char init_shuffle(unsigned char retain_old_pos) {
	unsigned short i;
	FILINFO old_file_info;
	unsigned short old_file_index = 0;
	const char* p;

	order_num_files = 0;
	memcpy(&old_file_info, &cur_file_info, sizeof(old_file_info));

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( !strcmp(cur_file_info.fname, old_file_info.fname) )
				old_file_index = order_num_files;
			if( ++order_num_files == SORT_ENTRIES )
				break;
		}
	} while( cur_pos.sect != 0 );

	seed_gerhard_random(GetRandomSeed());
	for( i = 0; i < order_num_files; ++i ) {
		insert_shuffle_index(get_gerhard_random()%(i+1), i, i);
	}

	if( order_num_files == 0 ) {
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		order_pos = retain_old_pos ? find_file_index(old_file_index) : 0;
	}

	if( playback_state == stopped )
		return goto_file_index(order[order_pos]);
	else
		return last_playback_error;
}

typedef int (*compfunc_t) (const void*, const void*);

static unsigned char init_sort(unsigned char retain_old_pos) {
	unsigned short i;
	unsigned char* sort_buffer;
	unsigned char* dest;
	const char* p;
	FILINFO old_file_info;
	unsigned short old_file_index = 0;

	order_num_files = 0;
	memcpy(&old_file_info, &cur_file_info, sizeof(old_file_info));

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	dest = sort_buffer = dac_get_buffer(); // DAC isn't running while we're doing this so we can use its buffer to sort the file names
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( !strcmp(cur_file_info.fname, old_file_info.fname) )
				old_file_index = order_num_files;
			strncpy((char*)dest, cur_file_info.lfname, SORT_ENTRY_SIZE-2);
			dest[SORT_ENTRY_SIZE-2] = '\0';
			dest[SORT_ENTRY_SIZE-1] = order_num_files;
			dest += SORT_ENTRY_SIZE;
			if( ++order_num_files == SORT_ENTRIES )
				break;
		}
	} while( cur_pos.sect != 0 );

	qsort(sort_buffer, order_num_files, SORT_ENTRY_SIZE, (compfunc_t)&strcmp);

	dest = sort_buffer+SORT_ENTRY_SIZE-1;
	for( i = 0; i < order_num_files; ++i ) {
		order[i] = dest[0];
		dest += SORT_ENTRY_SIZE;
	}

	memset(sort_buffer, 0, SORT_ENTRY_SIZE*SORT_ENTRIES);

	if( order_num_files == 0 ) {
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		order_pos = retain_old_pos ? find_file_index(old_file_index) : 0;
	}

	if( playback_state == stopped )
		return goto_file_index(order[order_pos]);
	else
		return last_playback_error;
}

static unsigned char update_playback_order(unsigned char retain_old_pos) {
	if( desired_playback_order != current_playback_order ) {
		current_playback_order = desired_playback_order;
		if( desired_playback_order == shuffle )
			return init_shuffle(retain_old_pos);
		else if( desired_playback_order == sorted )
			return init_sort(retain_old_pos);
	}
	return last_playback_error;
}

static unsigned char goto_next_wav_file(unsigned char retain_old_pos) {
	unsigned char found = 0;

	if( update_playback_order(retain_old_pos) != ok )
		return last_playback_error;
	if( !retain_old_pos )
		order_pos = SORT_ENTRIES-1;

try_next_file:
	if( current_playback_order != directory ) {
		if( ++order_pos >= order_num_files ) {
			if( cur_folder_level > 0 ) {
				Playback_Up_Folder();
				return goto_next_wav_file(true);
			} else {
				order_pos = 0;
			}
		}
		if( goto_file_index(order[order_pos]) )
			return last_playback_error;
		else
			goto try_open;
	}

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			found = 1;
			break;
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		if( cur_folder_level > 0 && Playback_Up_Folder() == ok )
			return goto_next_wav_file(retain_old_pos);
		else
			return goto_first_wav_file();
/*
		playback_state = error;
		last_playback_error = no_wav_files;
		return last_playback_error;
*/
	} else {
		unsigned char init_folder_level;
	try_open:
		init_folder_level = cur_folder_level;
		if( open_wav_file(false, false) == no_wav_files && (current_playback_order != directory || order_pos+1 < order_num_files) ) {
			while( cur_folder_level > init_folder_level )
				if( Playback_Up_Folder() != ok )
					return last_playback_error;
			if( update_playback_order(retain_old_pos) != ok )
				return last_playback_error;
			goto try_next_file;
		}
		return last_playback_error;
	}
}

static unsigned char goto_first_wav_file() {
	if( current_playback_order == directory && f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	return goto_next_wav_file(0);
}

unsigned char find_last_file() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_lfname[_MAX_LFN+1];
	bool found = false;

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}
	strcpy(old_lfname, cur_file_info.lfname);
	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			found = true;
			memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
			memcpy(&last_pos, &cur_pos, sizeof(last_pos));
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		playback_state = error;
		last_playback_error = no_wav_files;
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		return 1;
	}
}

unsigned char find_last_dir() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_lfname[_MAX_LFN+1];
	bool found = false;

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}
	strcpy(old_lfname, cur_file_info.lfname);
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( ((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0' ) {
			found = true;
			memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
			memcpy(&last_pos, &cur_pos, sizeof(last_pos));
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		return 1;
	}
}

unsigned char find_prev_dir() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_lfname[_MAX_LFN+1], init_fname[13];
	bool found = false;

	memcpy(&init_fname, &cur_file_info.fname, sizeof(init_fname));
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}
	strcpy(old_lfname, cur_file_info.lfname);
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( !memcmp(cur_file_info.fname, init_fname, sizeof(init_fname)) )
			break;
		if( ((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0' ) {
			found = true;
			memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
			memcpy(&last_pos, &cur_pos, sizeof(last_pos));
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		return 1;
	}
}

unsigned char goto_prev_wav_file(bool bGotoFirstInDir);

static unsigned char goto_last_wav_file() {
	unsigned char init_folder_level;

	if( update_playback_order(true) )
		return last_playback_error;
	if( current_playback_order == directory ) {
		if( !find_last_file() )
			return last_playback_error;
	} else {
        order_pos = order_num_files-1;
		if( goto_file_index(order[order_pos]) != ok )
			return last_playback_error;
	}

	init_folder_level = cur_folder_level;
	if( open_wav_file(true, true) == no_wav_files && (current_playback_order != directory || order_pos > 0) ) {
		while( cur_folder_level > init_folder_level )
			if( Playback_Up_Folder() != ok )
				return last_playback_error;
		if( update_playback_order(true) != ok )
			return last_playback_error;
		return goto_prev_wav_file(0);
    }      
	return last_playback_error;
}

static unsigned char goto_last_dir_or_first_wav_file() {
	if( update_playback_order(true) )
		return last_playback_error;
	if( current_playback_order == directory ) {
		if( find_last_dir() ) {
			unsigned char init_folder_level = cur_folder_level;
			while( open_wav_file(false, true) == no_wav_files ) {
				while( cur_folder_level > init_folder_level )
					if( Playback_Up_Folder() != ok )
						return last_playback_error;
				if( update_playback_order(true) != ok )
					return last_playback_error;
				if( find_prev_dir() != ok )
					return last_playback_error;
			}
		}
		if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}
		return goto_next_wav_file(0);
	} else {
        for( order_pos = order_num_files-1; order_pos < order_num_files; --order_pos ) {
			if( goto_file_index(order[order_pos]) != ok )
				return last_playback_error;
			if( (cur_file_info.fattrib&AM_DIR) ) {
				if( open_wav_file(false, true) != no_wav_files )
					return last_playback_error;
			}
		}
		return goto_first_wav_file();
	}
}

unsigned char find_prev_file() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_fname[13];
	char old_lfname[_MAX_LFN+1];
	unsigned char first_file = 1;
	unsigned char found = 0;

	strcpy(old_fname, cur_file_info.fname);
	strcpy(old_lfname, cur_file_info.lfname);
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return 0;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			found = 1;
			if( !strcmp(cur_file_info.fname, old_fname) && !first_file ) {
				break;
			} else {
				memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
				memcpy(&last_pos, &cur_pos, sizeof(last_pos));
				first_file = 0;
			}
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		playback_state = error;
		last_playback_error = fat_mount_failed;
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		strcpy(cur_file_info.lfname, old_lfname);
		return 1;
	}
}

unsigned char goto_prev_wav_file(bool bGotoFirstInDir) {
	unsigned char init_folder_level;

try_prev_file:
	if( update_playback_order(true) != ok )
		return last_playback_error;

	if( current_playback_order != directory ) {
		if( --order_pos >= order_num_files ) {
			if( cur_folder_level > 0 ) {
				Playback_Up_Folder();
				return goto_prev_wav_file(bGotoFirstInDir);
			} else {
				order_pos = order_num_files-1;
			}
		}

		if( goto_file_index(order[order_pos]) )
			return last_playback_error;
		// otherwise, fall through
	} else {
		if( !find_prev_file() )
			return goto_last_wav_file();
		// otherwise, fall through
	}
	init_folder_level = cur_folder_level;
	if( open_wav_file(!bGotoFirstInDir, true) == no_wav_files && (current_playback_order != directory || order_pos > 0) ) {
		while( cur_folder_level > init_folder_level )
			if( Playback_Up_Folder() != ok )
				return last_playback_error;
		goto try_prev_file;
	}
	return last_playback_error;
}

extern unsigned char memoryCardSystemUp;

unsigned char Playback_Setup() {
	unsigned char i;

	for( i = 0; i < INIT_ATTEMPTS; ++i ) {
		unsigned char result;
		Timer_Delay_ms_escape(INIT_DELAY, &memoryCardSystemUp, 0);
		result = disk_initialize(0);
		if( result != STA_NOINIT )
			break;
	}
	if( i == INIT_ATTEMPTS ) {
		playback_state = error;
		switch(cardInfo.ERROR) {
//		case ERROR_NOT_SDMMC_CARD:
		default:
			last_playback_error = sd_card_invalid_response;
			break;
		case ERROR_BAD_VOLTAGE_RANGE:
			last_playback_error = sd_card_wrong_voltage;
			break;
		case ERROR_SDMMC_CARD_TIMEOUT:
			last_playback_error = sd_card_timeout;
			break;
		}
		return last_playback_error;
	}
	if( f_mount(0, &fs) != FR_OK ) {
		playback_state = error;
		last_playback_error = fat_mount_failed;
		return last_playback_error;
	}
	current_playback_order = directory;
	play_just_one = 0;
    return goto_first_wav_file();
}

unsigned char Playback_Start() {
	if( playback_state == stopped || playback_state == paused ) {
		dac_pause(false);
		playback_state = playing;
	}
	play_just_one = 0;
	muted = 0;
	return last_playback_error;
}

unsigned char Playback_Reset() {
	playback_state = reset;
	last_playback_error = ok;
	dac_pause(true);
	play_just_one = 0;
	muted = 0;
	cur_folder_level = 0;
	strcpy(cur_path, "0:\\");
	return last_playback_error;
}

unsigned char Playback_Stop() {
	if( playback_state == playing || playback_state == paused ) {
		if( playback_state == playing )
			dac_pause(true);
		if( f_lseek(&cur_file, wav_data_start) != FR_OK ) {
			playback_state = error;
			last_playback_error = invalid_wav_file;
		} else {
			wav_data_align = 512 - ((512 - (wav_data_start&511)) & ~(wav_bytes_per_sample - 1));
			wav_bytes_remaining = wav_data_size;
			playback_state = stopped;
		}
	} else if( playback_state == stopped ) {
		return goto_first_wav_file();
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char Playback_Pause() {
	if( playback_state == playing || playback_state == paused ) {
		dac_pause(dac_is_running());
		playback_state = dac_is_running() ? playing : paused;
	}
	return last_playback_error;
}

static unsigned char Playback_Next_Track_int(unsigned char clear_buffer, unsigned char loop) {
	if( playback_state == playing && clear_buffer )
		dac_clear_buffer();
	if( playback_state != error ) {
		playback_state_t old_playback_state = playback_state;
		if( goto_next_wav_file(true) ) {
			if( last_playback_error == no_wav_files && current_playback_order == directory ) {
				if( !loop )
					return Playback_Reset();
				last_playback_error = ok;
				if( goto_first_wav_file() ) {
					dac_pause(true);
					return last_playback_error;
				}
			} else {
				dac_pause(true);
				return last_playback_error;
			}
		} else if( current_playback_order != directory && order_pos == 0 && !loop ) {
			return Playback_Reset();
		}
		playback_state = old_playback_state;
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char Playback_Next_Track(unsigned char clear_buffer) {
	return Playback_Next_Track_int(clear_buffer, 1);
}

unsigned char Playback_Prev_Track(unsigned char clear_buffer) {
	if( playback_state == playing && clear_buffer )
		dac_clear_buffer();
	if( playback_state != error ) {
		playback_state_t old_playback_state = playback_state;
		if( goto_prev_wav_file(false) ) {
			dac_pause(true);
			return last_playback_error;
		}
		playback_state = old_playback_state;
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char Playback_Forward() {
	if( playback_state == playing ) { // calculate number of samples in 10 seconds
		unsigned long seek_distance = (unsigned long)wav_sample_rate * (unsigned long)wav_bytes_per_sample * 10;
		seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
		if( wav_bytes_remaining > seek_distance ) {
			wav_bytes_remaining -= seek_distance;
			if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
				dac_pause(true);
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
		}
	}
	return last_playback_error;
}

unsigned char Playback_Back() {
	if( playback_state == playing ) { // calculate number of samples in 10 seconds
		unsigned long seek_distance = (unsigned long)wav_sample_rate * (unsigned long)wav_bytes_per_sample * 10;
		seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
		if( wav_bytes_remaining < wav_data_size ) {
			wav_bytes_remaining += seek_distance;
			if( wav_bytes_remaining > wav_data_size )
				wav_bytes_remaining = wav_data_size;
			if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
				dac_pause(true);
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
		}
	}
	return last_playback_error;
}
/*
unsigned char playback_play_single_file(unsigned char index) {
	if( update_playback_order(true) )
		return last_playback_error;

	if( playback_state == playing )
		pause_dac(true);

	if( current_playback_order == directory ) {
		unsigned char found = 0;

		if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}

		do {
			const char* p;
			if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			p = strrchr(cur_file_info.fname, '.');
			if( p && is_playable_ext(p) ) {
				if( index-- == 0 ) {
					found = 1;
					break;
				}
			}
		} while( cur_pos.sect != 0 );

		if( !found ) {
			playback_state = error;
			last_playback_error = no_wav_files;
			return last_playback_error;
		} else {
			if( open_wav_file(false, false) )
				return last_playback_error;
		}
	} else {
		if( index >= order_num_files ) {
			if( playback_state != stopped ) {
				playback_stop();
			}
			return last_playback_error;
		}
		order_pos = index-1;
		if( goto_next_wav_file(true) ) {
			if( last_playback_error == no_wav_files && current_playback_order == directory ) {
				last_playback_error = ok;
				if( goto_first_wav_file() ) {
					reset_sequencer(0);
					return last_playback_error;
				}
			} else {
				return last_playback_error;
			}
		}
	}

	if( playback_state == playing ) {
		pause_dac(true);
		return last_playback_error;
	} else {
		return playback_start();
	}
}
*/

unsigned char playback_volume_up() {
	if( g_volume < 8 )
		++g_volume;
	else
		g_volume = g_volume + (g_volume>>3);

	if( g_volume > 256 )
		g_volume = 256;
	muted = 0;
	return last_playback_error;
}

unsigned char playback_volume_down() {
	if( g_volume <= 8 && g_volume > 0 )
		--g_volume;
	else if( g_volume > 8 )
		g_volume -= g_volume>>3;
	muted = 0;
	return last_playback_error;
}

unsigned char playback_volume_set(unsigned char percent) {
	g_volume = percent * 256 / 100;
	return last_playback_error;
}

unsigned char playback_toggle_mute() {
	muted = !muted;
	return last_playback_error;
}

playback_state_t Playback_Get_Status() {
	return playback_state;
}

playback_error_t Playback_Get_Error() {
	return last_playback_error;
}

playback_order_t Playback_Get_Order() {
	return desired_playback_order;
}

unsigned char Playback_Set_Order(playback_order_t order) {
	desired_playback_order = order;
	return last_playback_error;
}

static unsigned char play_file() {
	UINT read = 0;
	signed short wavbuf[256];
	unsigned short read_size = sizeof(wavbuf);
	unsigned short num_channels;

	do {
		if( transitioning_to_next_file ) {
        	memset(wavbuf, 0, sizeof(wavbuf));
			num_channels = 2;
		} else {
			num_channels = wav_num_channels;
			if( num_channels == 1 )
				read_size >>= 1;
			if( wav_data_align ) {
				if( wav_data_align + read_size >= 512 )
					read_size = 512 - wav_data_align;
				memset(wavbuf, 0, wav_data_align);
			}
			if( wav_dummy ) {
				memset(wavbuf, 0, read_size);
				read = read_size;
			} else if( f_read(&cur_file, ((BYTE*)wavbuf) + wav_data_align, read_size, &read) != FR_OK ) {
				dac_pause(true);
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			if( read > wav_bytes_remaining )
				read = wav_bytes_remaining;
			wav_bytes_remaining -= read;
			if( wav_data_align ) {
				wav_data_align += read;
				if( wav_data_align >= 512 )
					wav_data_align = 0;
			}
			if( read != read_size || wav_bytes_remaining == 0 ) {
				memset(((BYTE*)wavbuf)+read, 0, read_size-read);
				transitioning_to_next_file = DAC_BUFFER_PAGES;
			}
		}
		dac_write(wavbuf, num_channels, muted ? 0 : g_volume);
	} while( !DACBuffer_full );

	return 0;
}

unsigned char Playback_Process() {
	if( playback_state == playing ) {
		unsigned char num_channels;

		do {
			if( play_file(&num_channels) )
				return last_playback_error;
	
			if( transitioning_to_next_file && --transitioning_to_next_file == 0 ) {
				if( play_just_one ) {
					play_just_one = 0;
					if( Playback_Stop() )
						return last_playback_error;
					return goto_first_wav_file();
				} else {
					return Playback_Next_Track_int(0, !dont_repeat_all);
				}
			}
		} while( !DACBuffer_full );
	}
	return last_playback_error;
}


unsigned char Playback_Get_Folder_Level() {
  return cur_folder_level;
}

unsigned char Playback_Next_Folder() {
	if( cur_folder_level > 0 ) {
		Playback_Up_Folder();
		return goto_next_wav_file(true);
	}
	return last_playback_error;
}

unsigned char Playback_Prev_Folder() {
	if( cur_folder_level > 0 ) {
		Playback_Up_Folder();
		return goto_prev_wav_file(true);
	}
	return last_playback_error;
}

unsigned char Playback_Up_Folder() {
	last_playback_error = ok;

	if( cur_folder_level == 0 ) {
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		char* bs = strrchr(cur_path, '\\');
		char oldchar = '\0';
		if( bs == cur_path+2 ) {
			oldchar = bs[1];
			bs[1] = '\0';
		} else {
			bs[0] = '\0';
		}

		if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}
		if( oldchar )
			bs[1] = oldchar;
		do {
			if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
				if( oldchar )
					bs[1] = '\0';
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			if( (cur_file_info.fattrib&AM_DIR) && !strcmp(cur_file_info.fname, bs+1) )
				break;
		} while( cur_pos.sect != 0 );
		if( oldchar )
			bs[1] = '\0';

		current_playback_order = directory;
        --cur_folder_level;
	}
	return last_playback_error;
}
